<?php
/**
 * Orders
 *
 * Shows orders on the account page.
 *
 * This template can be overridden by copying it to yourtheme/woocommerce/myaccount/orders.php.
 *
 * HOWEVER, on occasion WooCommerce will need to update template files and you
 * (the theme developer) will need to copy the new files to your theme to
 * maintain compatibility. We try to do this as little as possible, but it does
 * happen. When this occurs the version of the template file will be bumped and
 * the readme will list any important changes.
 *
 * @see https://woocommerce.com/document/template-structure/
 * @package WooCommerce\Templates
 * @version 9.2.0
 */

defined( 'ABSPATH' ) || exit;


do_action( 'woocommerce_before_account_orders', $has_orders ); 

$pagination_mid_size = 2;
$max_element_per_page = 10;
add_filter('woocommerce_my_account_my_orders_query', function($order_query_info) use ($max_element_per_page){
	$order_query_info['limit'] = $max_element_per_page; 
	return $order_query_info;
}, PHP_INT_MAX);

add_filter('woocommerce_account_orders_columns', function($columns){
	$columns = array(
		'order-number'  => __( 'Order', 'woocommerce' ),
		'order-description' => __( 'Description', 'tw-commerce' ),
		'order-date'    => __( 'Date', 'woocommerce' ),
		'order-status'  => __( 'Status', 'woocommerce' ),
		'order-total'   => __( 'Total', 'woocommerce' ),
		'order-actions' => __( 'Actions', 'woocommerce' )
	);

	return $columns;
}, PHP_INT_MAX, 2);
?>

<?php if ( $has_orders ) : ?>

	<table class="woocommerce-orders-table woocommerce-MyAccount-orders shop_table shop_table_responsive my_account_orders account-orders-table">
		<thead>
			<tr>
				<?php foreach ( wc_get_account_orders_columns() as $column_id => $column_name ) : ?>
					<th scope="col" class="woocommerce-orders-table__header woocommerce-orders-table__header-<?php echo esc_attr( $column_id ); ?>"><span class="nobr"><?php echo esc_html( $column_name ); ?></span></th>
				<?php endforeach; ?>
			</tr>
		</thead>

		<tbody>
			<?php
			foreach ( $customer_orders->orders as $customer_order ) {
				$order      = wc_get_order( $customer_order ); // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
				$item_count = $order->get_item_count() - $order->get_item_count_refunded();

                $order_products = $order->get_items('line_item');
                $product_names = null;
                $_product_image_url = null;
                $most_extensive_product = null;
                foreach($order_products as $item) {
                    $product = $item->get_product();

                    if($product->get_price() > $most_extensive_product || $most_extensive_product === null) {
                        $most_extensive_product = $product->get_price();
                    }
                }

                if($most_extensive_product !== null) {
                    if(!empty($product->get_image_id())) {
                        $_product_image_url = wp_get_attachment_image_url($product->get_image_id(), array(60, 60));
                    } else {
                        $_product_image_url = wc_placeholder_img_src('100');
                    }

					$product_names = $product->get_name();
					if(count($order_products) > 1) {
						$product_names .= " " . sprintf(__("et %d autres", "tw-commerce"), count($order_products));
					}
                }

				?>
				<tr class="woocommerce-orders-table__row woocommerce-orders-table__row--status-<?php echo esc_attr( $order->get_status() ); ?> order">
					<?php foreach ( wc_get_account_orders_columns() as $column_id => $column_name ) :
						$is_order_number = 'order-number' === $column_id;
					?>
						<?php if ( $is_order_number ) : ?>
							<th class="woocommerce-orders-table__cell woocommerce-orders-table__cell-<?php echo esc_attr( $column_id ); ?>" data-title="<?php echo esc_attr( $column_name ); ?>" scope="row">
						<?php else : ?>
							<td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-<?php echo esc_attr( $column_id ); ?>" data-title="<?php echo esc_attr( $column_name ); ?>">
						<?php endif; ?>

							<?php if ( has_action( 'woocommerce_my_account_my_orders_column_' . $column_id ) ) : ?>
								<?php do_action( 'woocommerce_my_account_my_orders_column_' . $column_id, $order ); ?>

							<?php elseif ( $is_order_number ) : ?>
								<?php /* translators: %s: the order number, usually accompanied by a leading # */ ?>
								<a href="<?php echo esc_url( $order->get_view_order_url() ); ?>" aria-label="<?php echo esc_attr( sprintf( __( 'View order number %s', 'woocommerce' ), $order->get_order_number() ) ); ?>">
									<?php echo esc_html( _x( '#', 'hash before order number', 'woocommerce' ) . $order->get_order_number() ); ?>
								</a>

							<?php elseif ('order-description' === $column_id) : ?>
								<div style="display:flex;">
									<img style="max-width:100px;" src="<?php echo $_product_image_url; ?>" />
									<span style="align-self:center;">
										<?php if($product_names !== null) {
											echo $product_names;	
										} ?>
									</span>
								</div>
							<?php elseif ( 'order-date' === $column_id ) : ?>
								<time datetime="<?php echo esc_attr( $order->get_date_created()->date( 'c' ) ); ?>"><?php echo esc_html( wc_format_datetime( $order->get_date_created() ) ); ?></time>

							<?php elseif ( 'order-status' === $column_id ) : ?>
								<?php echo esc_html( wc_get_order_status_name( $order->get_status() ) ); ?>

							<?php elseif ( 'order-total' === $column_id ) : ?>
								<?php
								/* translators: 1: formatted order total 2: total order items */
								echo wp_kses_post( sprintf( _n( '%1$s for %2$s item', '%1$s for %2$s items', $item_count, 'woocommerce' ), $order->get_formatted_order_total(), $item_count ) );
								?>

							<?php elseif ( 'order-actions' === $column_id ) : ?>
								<?php
								$actions = wc_get_account_orders_actions( $order );

								if ( ! empty( $actions ) ) {
									foreach ( $actions as $key => $action ) { // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
										/* translators: %s: order number */
										echo '<a href="' . esc_url( $action['url'] ) . '" class="woocommerce-button' . esc_attr( $wp_button_class ) . ' button ' . sanitize_html_class( $key ) . '" aria-label="' . esc_attr( sprintf( __( 'View order number %s', 'woocommerce' ), $order->get_order_number() ) ) . '">' . esc_html( $action['name'] ) . '</a>';
									}
								}
								?>
							<?php endif; ?>

						<?php if ( $is_order_number ) : ?>
							</th>
						<?php else : ?>
							</td>
						<?php endif; ?>
					<?php endforeach; ?>
				</tr>
				<?php
			}
			?>
		</tbody>
	</table>

	<?php do_action( 'woocommerce_before_account_orders_pagination' ); ?>

	<?php
		$nbr_pages = (int)($customer_orders->total / $max_element_per_page);
		if($customer_orders->total % $max_element_per_page > 0) {
			$nbr_pages = $nbr_pages + 1;
		}
	?>

	<?php if ( $nbr_pages > 0) : 
		$has_dot_before = false;
		$has_dot_after = false;
	?>
		<div class="woocommerce-pagination woocommerce-pagination--without-numbers woocommerce-Pagination">
			<?php if ( $current_page > 1 ) : ?>
				<a class="woocommerce-button woocommerce-button--previous woocommerce-Button woocommerce-Button--previous button<?php echo esc_attr( $wp_button_class ); ?>" href="<?php echo esc_url( wc_get_endpoint_url( 'orders', $current_page - 1 ) ); ?>"><?php esc_html_e( 'Previous', 'woocommerce' ); ?></a>
			<?php endif; ?>

			<?php for($i=1;$i<=$nbr_pages;$i++) { 
				if($i <= 1 || ($i >= $current_page - $pagination_mid_size && $i <= $current_page + $pagination_mid_size) || $i > $nbr_pages - 1) { ?>
					<a class="woocommerce-button woocommerce-pagination-link <?php if ($i == $current_page) { echo "current-link"; } ?>" href="<?php echo esc_url( wc_get_endpoint_url('orders', $i )); ?>"><?php echo $i; ?></a>
				<?php }
				
				if($i >= ($current_page - $pagination_mid_size) && $i < $current_page  && $has_dot_before == false) { ?>
					<a class="woocommerce-button woocommerce-pagination-link">...</a>
					<?php $has_dot_before = true; ?>
				<?php } else if ($i <= ($current_page + $pagination_mid_size) && $i > $current_page && $has_dot_after == false) { ?>
					<a class="woocommerce-button woocommerce-pagination-link">...</a>
					<?php $has_dot_after = true; ?>
				<?php } ?>
			<?php } ?>

			<?php if ( $current_page < $nbr_pages ) : ?>
				<a class="woocommerce-button woocommerce-button--next woocommerce-Button woocommerce-Button--next button<?php echo esc_attr( $wp_button_class ); ?>" href="<?php echo esc_url( wc_get_endpoint_url( 'orders', $current_page + 1 ) ); ?>"><?php esc_html_e( 'Next', 'woocommerce' ); ?></a>
			<?php endif; ?>
		</div>
	<?php endif; ?>
<?php else : ?>

	<?php wc_print_notice( esc_html__( 'No order has been made yet.', 'woocommerce' ) . ' <a class="woocommerce-Button wc-forward button' . esc_attr( $wp_button_class ) . '" href="' . esc_url( apply_filters( 'woocommerce_return_to_shop_redirect', wc_get_page_permalink( 'shop' ) ) ) . '">' . esc_html__( 'Browse products', 'woocommerce' ) . '</a>', 'notice' ); // phpcs:ignore WooCommerce.Commenting.CommentHooks.MissingHookComment ?>

<?php endif; ?>

<?php do_action( 'woocommerce_after_account_orders', $has_orders ); ?>