<?php
/**
 * Tests the Style Engine Processor class.
 *
 * @package WordPress
 * @subpackage StyleEngine
 * @since 6.1.0
 *
 * @group style-engine
 */

/**
 * Tests for compiling and rendering styles from a store of CSS rules.
 *
 * @coversDefaultClass WP_Style_Engine_Processor
 */
class Tests_Style_Engine_wpStyleEngineProcessor extends WP_UnitTestCase {
	/**
	 * Tests adding rules and returning compiled CSS rules.
	 *
	 * @ticket 56467
	 *
	 * @covers ::add_rules
	 * @covers ::get_css
	 */
	public function test_should_return_rules_as_compiled_css() {
		$a_nice_css_rule = new WP_Style_Engine_CSS_Rule( '.a-nice-rule' );
		$a_nice_css_rule->add_declarations(
			array(
				'color'            => 'var(--nice-color)',
				'background-color' => 'purple',
			)
		);
		$a_nicer_css_rule = new WP_Style_Engine_CSS_Rule( '.a-nicer-rule' );
		$a_nicer_css_rule->add_declarations(
			array(
				'font-family'      => 'Nice sans',
				'font-size'        => '1em',
				'background-color' => 'purple',
			)
		);
		$a_nice_processor = new WP_Style_Engine_Processor();
		$a_nice_processor->add_rules( array( $a_nice_css_rule, $a_nicer_css_rule ) );

		$this->assertSame(
			'.a-nice-rule{color:var(--nice-color);background-color:purple;}.a-nicer-rule{font-family:Nice sans;font-size:1em;background-color:purple;}',
			$a_nice_processor->get_css( array( 'prettify' => false ) )
		);
	}

	/**
	 * Tests adding nested rules with at-rules and returning compiled CSS rules.
	 *
	 * @ticket 61099
	 *
	 * @covers ::add_rules
	 * @covers ::get_css
	 */
	public function test_should_return_nested_rules_as_compiled_css() {
		$a_nice_css_rule = new WP_Style_Engine_CSS_Rule( '.a-nice-rule' );
		$a_nice_css_rule->add_declarations(
			array(
				'color'            => 'var(--nice-color)',
				'background-color' => 'purple',
			)
		);
		$a_nice_css_rule->set_rules_group( '@media (min-width: 80rem)' );

		$a_nicer_css_rule = new WP_Style_Engine_CSS_Rule( '.a-nicer-rule' );
		$a_nicer_css_rule->add_declarations(
			array(
				'font-family'      => 'Nice sans',
				'font-size'        => '1em',
				'background-color' => 'purple',
			)
		);
		$a_nicer_css_rule->set_rules_group( '@layer nicety' );

		$a_nice_processor = new WP_Style_Engine_Processor();
		$a_nice_processor->add_rules( array( $a_nice_css_rule, $a_nicer_css_rule ) );

		$this->assertSame(
			'@media (min-width: 80rem){.a-nice-rule{color:var(--nice-color);background-color:purple;}}@layer nicety{.a-nicer-rule{font-family:Nice sans;font-size:1em;background-color:purple;}}',
			$a_nice_processor->get_css( array( 'prettify' => false ) )
		);
	}

	/**
	 * Tests compiling CSS rules and formatting them with new lines and indents.
	 *
	 * @ticket 56467
	 *
	 * @covers ::get_css
	 */
	public function test_should_return_prettified_css_rules() {
		$a_wonderful_css_rule = new WP_Style_Engine_CSS_Rule( '.a-wonderful-rule' );
		$a_wonderful_css_rule->add_declarations(
			array(
				'color'            => 'var(--wonderful-color)',
				'background-color' => 'orange',
			)
		);
		$a_very_wonderful_css_rule = new WP_Style_Engine_CSS_Rule( '.a-very_wonderful-rule' );
		$a_very_wonderful_css_rule->add_declarations(
			array(
				'color'            => 'var(--wonderful-color)',
				'background-color' => 'orange',
			)
		);
		$a_more_wonderful_css_rule = new WP_Style_Engine_CSS_Rule( '.a-more-wonderful-rule' );
		$a_more_wonderful_css_rule->add_declarations(
			array(
				'font-family'      => 'Wonderful sans',
				'font-size'        => '1em',
				'background-color' => 'orange',
			)
		);
		$a_wonderful_processor = new WP_Style_Engine_Processor();
		$a_wonderful_processor->add_rules( array( $a_wonderful_css_rule, $a_very_wonderful_css_rule, $a_more_wonderful_css_rule ) );

		$expected = '.a-wonderful-rule {
	color: var(--wonderful-color);
	background-color: orange;
}
.a-very_wonderful-rule {
	color: var(--wonderful-color);
	background-color: orange;
}
.a-more-wonderful-rule {
	font-family: Wonderful sans;
	font-size: 1em;
	background-color: orange;
}
';
		$this->assertSameIgnoreEOL(
			$expected,
			$a_wonderful_processor->get_css( array( 'prettify' => true ) )
		);
	}

	/**
	 * Tests compiling nested CSS rules and formatting them with new lines and indents.
	 *
	 * @ticket 61099
	 *
	 * @covers ::get_css
	 */
	public function test_should_return_prettified_nested_css_rules() {
		$a_wonderful_css_rule = new WP_Style_Engine_CSS_Rule( '.a-wonderful-rule' );
		$a_wonderful_css_rule->add_declarations(
			array(
				'color'            => 'var(--wonderful-color)',
				'background-color' => 'orange',
			)
		);
		$a_wonderful_css_rule->set_rules_group( '@media (min-width: 80rem)' );

		$a_very_wonderful_css_rule = new WP_Style_Engine_CSS_Rule( '.a-very_wonderful-rule' );
		$a_very_wonderful_css_rule->add_declarations(
			array(
				'color'            => 'var(--wonderful-color)',
				'background-color' => 'orange',
			)
		);
		$a_very_wonderful_css_rule->set_rules_group( '@layer wonderfulness' );

		$a_wonderful_processor = new WP_Style_Engine_Processor();
		$a_wonderful_processor->add_rules( array( $a_wonderful_css_rule, $a_very_wonderful_css_rule ) );

		$expected = '@media (min-width: 80rem) {
	.a-wonderful-rule {
		color: var(--wonderful-color);
		background-color: orange;
	}
}
@layer wonderfulness {
	.a-very_wonderful-rule {
		color: var(--wonderful-color);
		background-color: orange;
	}
}
';
		$this->assertSame(
			$expected,
			$a_wonderful_processor->get_css( array( 'prettify' => true ) )
		);
	}

	/**
	 * Tests adding a store and compiling CSS rules from that store.
	 *
	 * @ticket 56467
	 *
	 * @covers ::add_store
	 */
	public function test_should_return_store_rules_as_css() {
		$a_nice_store = WP_Style_Engine_CSS_Rules_Store::get_store( 'nice' );
		$a_nice_store->add_rule( '.a-nice-rule' )->add_declarations(
			array(
				'color'            => 'var(--nice-color)',
				'background-color' => 'purple',
			)
		);
		$a_nice_store->add_rule( '.a-nicer-rule' )->add_declarations(
			array(
				'font-family'      => 'Nice sans',
				'font-size'        => '1em',
				'background-color' => 'purple',
			)
		);
		$a_nice_renderer = new WP_Style_Engine_Processor();
		$a_nice_renderer->add_store( $a_nice_store );

		$this->assertSame(
			'.a-nice-rule{color:var(--nice-color);background-color:purple;}.a-nicer-rule{font-family:Nice sans;font-size:1em;background-color:purple;}',
			$a_nice_renderer->get_css( array( 'prettify' => false ) )
		);
	}

	/**
	 * Tests that CSS declarations are merged and deduped in the final CSS rules output.
	 *
	 * @ticket 56467
	 *
	 * @covers ::add_rules
	 * @covers ::get_css
	 */
	public function test_should_dedupe_and_merge_css_declarations() {
		$an_excellent_rule      = new WP_Style_Engine_CSS_Rule( '.an-excellent-rule' );
		$an_excellent_processor = new WP_Style_Engine_Processor();
		$an_excellent_rule->add_declarations(
			array(
				'color'        => 'var(--excellent-color)',
				'border-style' => 'dotted',
			)
		);
		$an_excellent_processor->add_rules( $an_excellent_rule );

		$another_excellent_rule = new WP_Style_Engine_CSS_Rule( '.an-excellent-rule' );
		$another_excellent_rule->add_declarations(
			array(
				'color'        => 'var(--excellent-color)',
				'border-style' => 'dotted',
				'border-color' => 'brown',
			)
		);
		$an_excellent_processor->add_rules( $another_excellent_rule );

		$this->assertSame(
			'.an-excellent-rule{color:var(--excellent-color);border-style:dotted;border-color:brown;}',
			$an_excellent_processor->get_css( array( 'prettify' => false ) ),
			'Return value of get_css() does not match expectations with new, deduped and merged declarations.'
		);

		$yet_another_excellent_rule = new WP_Style_Engine_CSS_Rule( '.an-excellent-rule' );
		$yet_another_excellent_rule->add_declarations(
			array(
				'color'        => 'var(--excellent-color)',
				'border-style' => 'dashed',
				'border-width' => '2px',
			)
		);
		$an_excellent_processor->add_rules( $yet_another_excellent_rule );

		$this->assertSame(
			'.an-excellent-rule{color:var(--excellent-color);border-style:dashed;border-color:brown;border-width:2px;}',
			$an_excellent_processor->get_css( array( 'prettify' => false ) ),
			'Return value of get_css() does not match expectations with deduped and merged declarations.'
		);
	}

	/**
	 * Tests printing out 'unoptimized' CSS, that is, uncombined selectors and duplicate CSS rules.
	 *
	 * This is the default.
	 *
	 * @ticket 58811
	 * @ticket 56467
	 *
	 * @covers ::get_css
	 */
	public function test_should_not_optimize_css_output() {
		$a_sweet_rule = new WP_Style_Engine_CSS_Rule(
			'.a-sweet-rule',
			array(
				'color'            => 'var(--sweet-color)',
				'background-color' => 'purple',
			)
		);

		$a_sweeter_rule = new WP_Style_Engine_CSS_Rule(
			'#an-even-sweeter-rule > marquee',
			array(
				'color'            => 'var(--sweet-color)',
				'background-color' => 'purple',
			)
		);

		$the_sweetest_rule = new WP_Style_Engine_CSS_Rule(
			'.the-sweetest-rule-of-all a',
			array(
				'color'            => 'var(--sweet-color)',
				'background-color' => 'purple',
			)
		);

		$a_sweet_processor = new WP_Style_Engine_Processor();
		$a_sweet_processor->add_rules( array( $a_sweet_rule, $a_sweeter_rule, $the_sweetest_rule ) );

		$this->assertSame(
			'.a-sweet-rule{color:var(--sweet-color);background-color:purple;}#an-even-sweeter-rule > marquee{color:var(--sweet-color);background-color:purple;}.the-sweetest-rule-of-all a{color:var(--sweet-color);background-color:purple;}',
			$a_sweet_processor->get_css(
				array(
					'optimize' => false,
					'prettify' => false,
				)
			)
		);
	}

	/**
	 * Tests that 'optimized' CSS is output, that is, that duplicate CSS rules are combined under their corresponding selectors.
	 *
	 * @ticket 58811
	 * @ticket 56467
	 *
	 * @covers ::get_css
	 */
	public function test_should_not_optimize_css_output_by_default() {
		$a_sweet_rule = new WP_Style_Engine_CSS_Rule(
			'.a-sweet-rule',
			array(
				'color'            => 'var(--sweet-color)',
				'background-color' => 'purple',
			)
		);

		$a_sweeter_rule = new WP_Style_Engine_CSS_Rule(
			'#an-even-sweeter-rule > marquee',
			array(
				'color'            => 'var(--sweet-color)',
				'background-color' => 'purple',
			)
		);

		$a_sweet_processor = new WP_Style_Engine_Processor();
		$a_sweet_processor->add_rules( array( $a_sweet_rule, $a_sweeter_rule ) );

		$this->assertSame(
			'.a-sweet-rule{color:var(--sweet-color);background-color:purple;}#an-even-sweeter-rule > marquee{color:var(--sweet-color);background-color:purple;}',
			$a_sweet_processor->get_css( array( 'prettify' => false ) )
		);
	}

	/**
	 * Tests that incoming CSS rules are optimized and merged with existing CSS rules.
	 *
	 * @ticket 58811
	 * @ticket 56467
	 *
	 * @covers ::add_rules
	 */
	public function test_should_combine_previously_added_css_rules() {
		$a_lovely_processor = new WP_Style_Engine_Processor();
		$a_lovely_rule      = new WP_Style_Engine_CSS_Rule(
			'.a-lovely-rule',
			array(
				'border-color' => 'purple',
			)
		);
		$a_lovely_processor->add_rules( $a_lovely_rule );
		$a_lovelier_rule = new WP_Style_Engine_CSS_Rule(
			'.a-lovelier-rule',
			array(
				'border-color' => 'purple',
			)
		);
		$a_lovely_processor->add_rules( $a_lovelier_rule );

		$this->assertSame(
			'.a-lovely-rule,.a-lovelier-rule{border-color:purple;}',
			$a_lovely_processor->get_css(
				array(
					'prettify' => false,
					'optimize' => true,
				)
			),
			'Return value of get_css() does not match expectations when combining 2 CSS rules'
		);

		$a_most_lovely_rule = new WP_Style_Engine_CSS_Rule(
			'.a-most-lovely-rule',
			array(
				'border-color' => 'purple',
			)
		);
		$a_lovely_processor->add_rules( $a_most_lovely_rule );

		$a_perfectly_lovely_rule = new WP_Style_Engine_CSS_Rule(
			'.a-perfectly-lovely-rule',
			array(
				'border-color' => 'purple',
			)
		);
		$a_lovely_processor->add_rules( $a_perfectly_lovely_rule );

		$this->assertSame(
			'.a-lovely-rule,.a-lovelier-rule,.a-most-lovely-rule,.a-perfectly-lovely-rule{border-color:purple;}',
			$a_lovely_processor->get_css(
				array(
					'prettify' => false,
					'optimize' => true,
				)
			),
			'Return value of get_css() does not match expectations when combining 4 CSS rules'
		);
	}
}
