<?php

use Automattic\WooCommerce\Checkout\Helpers\ReserveStock;

/**
 * Problème gestion du backorder liit quand stock_gere_par_woocommerce
 *
 * @group hooks
 */
class Tests_StockManager extends WP_UnitTestCase {


	/**
	 * ATTENTION AU NIVEAU DE LA TEST SUITE SI WP_RUN_CORE_TESTS = 1 les post types sont resets entre chaque test
	 */
    public static function wpSetUpBeforeClass() {

		$pluginsToActivate = [
			dirname(__DIR__, 4) . '/woocommerce/woocommerce.php',
			dirname(__DIR__, 4) . '/tw-core/tw-core.php',
			dirname(__DIR__, 4) . '/tw-core/classes/tw_product_simple.php'    
		];
		foreach($pluginsToActivate as $plugin) {
			require_once $plugin;
		}

		require_once dirname(__DIR__, 1) . '/includes/tw_include/testStockHandler.php';
		do_action('init');
    }
	

	public function tear_down(){
		parent::tear_down();

		$testProducts = new WC_Product_Query([
			'name' => 'test_product'
		]);
		$testProducts->get_products();
		foreach($testProducts as $item) {
			$item->delete();
		}
		wc_delete_product_transients();
	}
	
	private function create_product($stockManagementMode, $quantityInStock, $backOrderLimit, $price = 50, $id = null) {
		$fakeProductId = $id !== null ? $id : 99998;
		$quantityInStock = $quantityInStock;
		$backOrderLimit = $backOrderLimit;

		$product = new TW_Product_Simple();
		$product->set_name("test_product");
		$product->set_category_ids([23]);
		$product->set_price($price);
		$product->set_regular_price($price);
		$product->set_stock_quantity($quantityInStock);
		if($quantityInStock > 0) {
			$product->set_stock_status('instock');
		} else if($quantityInStock < 0 && abs($quantityInStock) < $backOrderLimit){
			$product->set_stock_status('onbackorder');
		} else {
			$product->set_stock_status('outofstock');
		}

		$product->set_id($fakeProductId);
		$product->set_stock_management_data(array(
			'stock_management_mode' => $stockManagementMode,
			'stock_management_url' => ''
		));
		$product->set_backorder_limit($backOrderLimit);
		$product->set_supplier_data(array(
			'product_id' => $fakeProductId,
			'supplier_name' => 'test',
			'restocking_delay' => 5,
			'manage_restocking' => 0,
			'supplier_restocking_url' => null
		));
		
		TW_External_Stock_Manager::getInstance()->registerHandler(new Test_StockHandler($quantityInStock));
		TW_Factory::getInstance()->getClass(TW_External_Stock_Repository::class)->deleteStock($fakeProductId);
		return $product;
	}

	public function test_product_stock_managed_by_woocommerce() {
		$availableStock1 = null;
		$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 0, 0);
		remove_filter('woocommerce_get_availability_text', 'tw_stock_get_availability_text', PHP_INT_MAX);

		$availabilityMethod = new ReflectionMethod(WC_Product::class, 'get_availability_text');
		$availabilityMethod->setAccessible(true);
		$availableStock1 = $availabilityMethod->invoke($product);

		$availableStock2 = null;
		$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 0, 0);
		add_filter('woocommerce_get_availability_text', 'tw_stock_get_availability_text', PHP_INT_MAX, 2);

		$availabilityMethod = new ReflectionMethod(WC_Product::class, 'get_availability_text');
		$availabilityMethod->setAccessible(true);
		$availableStock2 = $availabilityMethod->invoke($product);
		return $this->assertSame($availableStock1, $availableStock2);
	}

	public function test_product_stock_visibility_outofstock() {
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,0);

		$availabilityMethod = new ReflectionMethod(WC_Product::class, 'get_availability_text');
		$availabilityMethod->setAccessible(true);
		return $this->assertSame($availabilityMethod->invoke($product), __('Out of stock', 'woocommerce'));
	}

	public function test_product_stock_visibility_backorder_allowed() {
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,3);

		$availabilityMethod = new ReflectionMethod(WC_Product::class, 'get_availability_text');
		$availabilityMethod->setAccessible(true);
		return $this->assertSame($availabilityMethod->invoke($product), __('Available on backorder', 'woocommerce'));
	}

	public function test_product_stock_visibility_backorder_notallowed() {
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', -3,2);

		$availabilityMethod = new ReflectionMethod(WC_Product::class, 'get_availability_text');
		$availabilityMethod->setAccessible(true);
		return $this->assertSame($availabilityMethod->invoke($product), __('Out of stock', 'woocommerce'));
	}

	public function test_product_button_visibility_outofstock() {
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,0);
		return $this->assertNotTrue($product->is_in_stock());
	}

	public function test_product_button_visibility_backorder_allowed() {
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,3);
		return $this->assertTrue($product->is_in_stock());
	}

	public function test_product_button_visibility_backorder_notallowed() {

		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', -3,2);
		return $this->assertNotTrue($product->is_in_stock());
	}

	public function test_product_button_visibility_managed_by_woocommerce() {
		$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 0,3);
		remove_filter('woocommerce_product_is_in_stock', 'tw_stock_product_is_in_stock', PHP_INT_MAX);
		$isInStockWithoutHook = $product->is_in_stock();

		$product2 = $this->create_product('dedicated_stock_managed_by_woocommerce', 0,0);
		add_filter('woocommerce_product_is_in_stock', 'tw_stock_product_is_in_stock', PHP_INT_MAX, 2);
		$isInStockWithHook = $product2->is_in_stock();

		return $this->assertSame($isInStockWithoutHook, $isInStockWithHook);
	}

	public function test_add_to_cart_item_without_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);

		$nbrItemInCart = 0;
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,0, "22.93", 0);
		$product->save();

		try {
			WC()->cart->add_to_cart($product->get_id(), 1);
			$nbrItemInCart = WC()->cart->get_cart();
		} catch (\Exception $ex) {}

		return $this->assertEquals(0,count($nbrItemInCart));
	}

	public function test_add_to_cart_item_with_backorder()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);
		
		$product = null;
		$nbrItemInCart = 0;
		try {
			$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0, 2, "53.50", 0);
			$product->save();

			WC()->cart->add_to_cart($product->get_id(), 1);
			foreach(WC()->cart->get_cart() as $item) {
				$nbrItemInCart = $nbrItemInCart + (int)$item['quantity'];
			}
		} catch (\Exception $ex) {}

		return $this->assertEquals(1, $nbrItemInCart);
	}
	
	public function test_add_to_cart_with_held_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);

		$product = null;
		$order = null;
		$nbrItemInCart = 0;

		try {
			$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 1, 0, "53.50", 0);
			$product->save();

			WC()->cart->add_to_cart($product->get_id(), 1);

			$order = new WC_Order();
			$order->set_customer_id(6);
			$order->set_currency('EUR');
			WC()->checkout()->create_order_line_items($order, WC()->cart);
			$order->save();

			wc_reserve_stock_for_order($order);
			WC()->cart->empty_cart();
			WC()->cart->add_to_cart($product->get_id(), 1);
			foreach(WC()->cart->get_cart() as $item) {
				$nbrItemInCart = $nbrItemInCart + (int)$item['quantity'];
			}
		} catch(\Exception $ex) {}

		if(!empty($order->get_id())){
			$order->delete();
		}

		return $this->assertEquals(0, $nbrItemInCart);
	}

	public function test_add_to_cart_with_insufficiant_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);

		$nbrItemInCart = 0;
		$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 2,0, "22.93", 0);
		$product->save();

		try {
			WC()->cart->add_to_cart($product->get_id(), 3);
			foreach(WC()->cart->get_cart() as $item) {
				$nbrItemInCart = $nbrItemInCart + (int)$item['quantity'];
			}
		} catch (\Exception $ex) {}
		
		return $this->assertEquals(2,$nbrItemInCart);
	}

	/**
	 * 	Process the checkout after button is pressed
	 *  2 methodes soit on appelle WC_Checkout->get_checkout_fields()
	 *  on remplit les valeurs $_POST et $_REQUEST
	 *  
	 *  on appelle process_checkout en ayant définit $_REQUEST ET $_POST
	 *  on vérifier que le résultat est un json qui contient le mot failure
	 * 
	 *  soit on trigger woocommerce_after_checkout_validation et et on vérifier qu'une erreur a bien été rajoutée dans la mesure ou on utilise pas
	 * wc_data on va pencher pour la deuxième solution
	 */
	public function test_cart_validation_item_without_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);
		remove_filter('woocommerce_add_to_cart_quantity','tw_add_to_cart_quantity', PHP_INT_MAX, 2);

		$nbrItemInCart = 0;
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 0,0, "22.93", 0);
		$product->save();

		$errors = new WP_Error();
		try {
			WC()->cart->add_to_cart($product->get_id(), 1);

			$data = [];
			do_action('woocommerce_after_checkout_validation', $data, $errors);
		} catch (\Exception $ex) {}

		$valid = WC()->cart->is_empty() || count($errors->errors) == 1;
		return $this->assertEquals($valid, true);
	}

	public function test_cart_validation_with_insufficiant_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);
		remove_filter('woocommerce_add_to_cart_quantity', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);

		$nbrItemInCart = 0;
		$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 2,0, "22.93", 0);
		$product->save();

		$errors = new WP_Error();
		try {
			WC()->cart->add_to_cart($product->get_id(), 3);

			$data = [];
			do_action('woocommerce_after_checkout_validation', $data, $errors);
		} catch (\Exception $ex) {}

		$valid = WC()->cart->is_empty() || count($errors->errors) == 1;
		return $this->assertEquals($valid, true);
	}

	public function test_cart_validation_with_backorder()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);
		remove_filter('woocommerce_add_to_cart_quantity','tw_add_to_cart_quantity', PHP_INT_MAX, 2);

		$nbrItemInCart = 0;
		$product = $this->create_product('dedicated_stock_not_managed_by_woocommerce', 1,2, "22.93", 0);
		$product->save();

		$errors = new WP_Error();
		try {
			WC()->cart->add_to_cart($product->get_id(), 2);

			$data = [];
			do_action('woocommerce_after_checkout_validation', $data, $errors);
		} catch (\Exception $ex) {}

		$valid = !WC()->cart->is_empty() || count($errors->errors) == 0;
		return $this->assertEquals($valid, true);
	}

	public function test_cart_validation_with_held_stock()
	{
		remove_action('woocommerce_after_product_object_save', 'tw_woocommerce_after_product_object_save', PHP_INT_MAX, 2);

		$product = null;
		$order = null;
		$nbrItemInCart = 0;

		$errors = new WP_Error();
		try {
			$product = $this->create_product('dedicated_stock_managed_by_woocommerce', 1, 0, "53.50", 0);
			$product->save();

			WC()->cart->add_to_cart($product->get_id(), 1);

			$order = new WC_Order();
			$order->set_customer_id(6);
			$order->set_currency('EUR');
			WC()->checkout()->create_order_line_items($order, WC()->cart);
			$order->save();

			wc_reserve_stock_for_order($order);
			WC()->cart->empty_cart();
			WC()->cart->add_to_cart($product->get_id(), 1);
			
			$data = [];
			do_action('woocommerce_after_checkout_validation', $data, $errors);
		} catch(\Exception $ex) {}

		if(!empty($order->get_id())){
			$order->delete();
		}

		$valid = !WC()->cart->is_empty() || count($errors->errors) == 0;
		return $this->assertEquals($valid, true);
	}
}