<?php
defined( 'ABSPATH' ) || exit;

class TW_Task_Repository {
    const FREQUENCY_ONCE = "ONCE";
    const FREQUENCY_DAILY = "DAILY";
    const FREQUENCY_WEEKLY = "WEEKLY";
    const FREQUENCY_MONTHLY = "MONTHLY";

    public function getExistingTasks($category, $name = null, DateTime $nextExecutionDateBefore = null, $offset = null, $limit = null)
    {
        global $wpdb;
        $stmt = "SELECT * FROM tw_task ";
        $stmt .= "WHERE task_category = " . esc_sql($category);

        if($name !== null) {
            $stmt .= " AND task_name = " . esc_sql($name);
        }

        if($nextExecutionDateBefore !== null) {
            $allowedDelay = 10 * 60;
            $stmt .= " AND next_execution_date IS NOT NULL AND TIMESTAMPDIFF(SECOND, next_execution_date, " . esc_sql($nextExecutionDateBefore->format('Y-m-d H:i:s')) . ") >= " . $allowedDelay;
        }

        if($offset !== null && $limit !== null) {
            $stmt .= " LIMIT " . $limit . " OFFSET " . $offset;
        }

        return $wpdb->get_results($stmt, ARRAY_A);
    }


    public function getNextExecutionDatetime($frequency, ?DateTime $from) {
        $from = $from === null ? new DateTime() : $from;
        $nextExecutionDate = clone($from);

        if($frequency == self::FREQUENCY_ONCE) {
            $nextExecutionDate = null;
        } else if($frequency == self::FREQUENCY_DAILY) {
            $nextExecutionDate = $nextExecutionDate->add(new DateInterval('P1D'));
        } else if($frequency == self::FREQUENCY_WEEKLY) {
            $nextExecutionDate = $nextExecutionDate->add(new DateInterval('P7D'));
        } else if($frequency == self::FREQUENCY_MONTHLY) {
            $currentMonth = (int)$from->format('m');
            $nextMonth = $currentMonth;
            $currentYear = (int)$from->format('Y');
            $nextYear = $currentYear;

            if($currentMonth < 12) {
                $nextMonth = $currentMonth + 1;
            } else {
                $nextMonth = $currentMonth + 1;
                $nextYear = $currentYear + 1;
            }

            $lastDayOfNextMonth = tw_get_last_day_of_month($nextMonth, $nextYear, 'd');
            if((int)$lastDayOfNextMonth >= (int)$from->format('d')) {
                $nextExecutionDate = $nextExecutionDate->setDate($nextYear, $nextMonth, (int)$from->format('d'));
            } else {
                $nextExecutionDate = $nextExecutionDate->setDate($nextYear, $nextMonth, (int)$lastDayOfNextMonth);
            }
        }

        return $nextExecutionDate;
    }

    public function insertTask($category, $name, $args = [], $frequency = "WEEKLY", ?DateTime $firstExecutionDate)
    {
        global $wpdb;

        if($firstExecutionDate === null) {
            $firstExecutionDate = new DateTime();
        }
        $args = json_encode($args);

        $nextExecutionDate = $this->getNextExecutionDatetime($frequency, $firstExecutionDate);
        if($nextExecutionDate !== null) {
            $nextExecutionDate = $nextExecutionDate->format('Y-m-d H:i:s');
        }

        $stmt = "INSERT INTO tw_task(task_category, task_name, task_args, task_frequency, next_execution_date)";
        $stmt .= "VALUES (" ;
        $stmt .= esc_sql($category) . ", ";
        $stmt .= esc_sql($name) . ", ";
        $stmt .= $args . ", ";
        $stmt .= esc_sql($frequency) . ", ";

        if($nextExecutionDate !== null) {
            $stmt .= esc_sql($nextExecutionDate);
        } else {
            $stmt .= "NULL";
        }
        $stmt .= ")";
        $wpdb->exec($stmt);
    }

    public function updateTask($category, $name, $args = [], $frequency = "WEEKLY", ?DateTime $executionDate, bool $success = true, $response = [])
    {
        global $wpdb;

        if($executionDate === null) {
            $executionDate = new DateTime();
        }
        $args = json_encode($args);
        $response = json_encode($response);
        
        $nextExecutionDate = $this->getNextExecutionDatetime($frequency, $executionDate);
        if($nextExecutionDate !== null) {
            $nextExecutionDate = $nextExecutionDate->format('Y-m-d H:i:s');
        }

        $stmt = "UPDATE tw_task SET task_args = " . $args;
        $stmt .= ", task_frequency = " . esc_sql($frequency);

        $stmt .= ", execution_date = " . esc_sql($executionDate->format('Y-m-d H:i:s'));

        if($nextExecutionDate !== null) {
            $stmt .= ", next_execution_date = " . esc_sql($nextExecutionDate);
        } else {
            $stmt .= ", next_execution_date = NULL";
        }

        if($success == true) {
            $stmt .= ", task_success = 1";
        } else {
            $stmt .= ", task_success = 0";
        }

        $stmt .= ", task_response = " . $response;

        $stmt .= " WHERE task_category = " . esc_sql($category) . " AND name = " . esc_sql($name);
        $wpdb->exec($stmt);
    }

    public function deleteTask($category, $name) {
        global $wpdb;
        $table ='tw_tasks';
        
        return $wpdb->delete($table, [
            'task_category' => $category,
            'task_name' => $name
        ]);
    }
}