<?php
defined( 'ABSPATH' ) || exit;

class TW_Shipping_Repository {

    protected $_cache = [
        'shippingDetail' => [],
        'shipping' => []
    ];

    public function getOrdersIdWithoutShipping($status = "wc-completed", $maxNbrDays = 3)
    {
        global $wpdb;
        $ordersIdWithoutShipping = [];

        $query = "SELECT * FROM wp_wc_orders WHERE status = '" . esc_sql($status) . "'";
        $query .= " AND DATEDIFF(CURDATE(), date_created_gmt) <= " . $maxNbrDays;
        $query .= " ORDER BY id";
        $query .= " LIMIT 5";

        $res = $wpdb->get_results(
            $query,
            ARRAY_A
        );

        foreach($res as $item) {
            $existingShippings = $this->getShippingsByOrderId($item['id'], true);
            if(empty($existingShippings)){
                $ordersIdWithoutShipping[] = $item['id']; 
            }
        }
        return $ordersIdWithoutShipping;
    }

    public function getShippedQuantityForOrderItem($order_id, $item_id, $forceReload = false) : int
    {
        $shippedQuantity = 0;

        $shippingDetails = $this->getOrderShippingDetails($order_id, $forceReload);
        foreach($shippingDetails as $detail) {
            if($detail['item_id'] == $item_id) {
                $shippedQuantity = $shippedQuantity + (int)$detail['item_quantity'];
            }
        }
        return $shippedQuantity;
    }

    public function getShippingByShippingId($shipping_id) : ?TW_Shipping
    {
        global $wpdb;
        $shipping = null;

        $query = "SELECT * FROM tw_shipping where expedition_id = " . esc_sql($shipping_id);
        $res = $wpdb->get_results(
            $query,
            ARRAY_A
        );
        if(!empty($res) && count($res) == 1){
            $shipping = new TW_Shipping($res[0]);
        }

        return $shipping;
    }

    public function getShippingDetailsByShippingId($shipping_id)
    {
        global $wpdb;
        $shippingDetails = [];

        $query = "SELECT twsd.* 
            FROM tw_shipping tws
            INNER JOIN tw_shipping_detail twsd ON (twsd.expedition_id = tws.expedition_id)
            WHERE tws.expedition_id = " . esc_sql($shipping_id);
            
        $res = $wpdb->get_results(
            $query,
            ARRAY_A
        );
        foreach($res as $item) {
            $shippingDetails[] = new TW_Shipping_Detail($item);
        }
        return $shippingDetails;
    }

    public function getShippingToRegenerate($maxNbrDays = 3)
    {
        global $wpdb;
        $shippings = [];
        $now = new DateTime();

        $shippingWaitingForCompletion = "SELECT ts.*
            FROM tw_shipping ts
            INNER JOIN tw_shipping ts2 ON (ts2.expedition_id = ts.expedition_id AND ts2.expedition_id IN (
                SELECT MAX(expedition_id)
                FROM tw_shipping ts3 
                WHERE ts3.order_id = ts.order_id
            )
        )";
        $res = $wpdb->get_results(
            $shippingWaitingForCompletion,
            ARRAY_A
        );

        foreach($res as $item) {
            $labelGenerated = $item['label_generated'];

            $generatedSince = null;
            if(!empty($item['label_generation_time'])){
                $lastGenerationTime = \DateTime::createFromFormat('Y-m-d H:i:s', $item['label_generation_time']);
                $lastGenerationTime->diff($now);
                $generatedSince = $lastGenerationTime->days;
            }

            if($labelGenerated == 0 && $generatedSince <= $maxNbrDays) {
                $item['shipping_details'] = $this->getShippingDetailsByExpeditionId($item['expedition_id']);
                $shippings[] = $item;
            }
        }
        return $shippings;
    }

    public function getShippingsByOrderId($order_id, $forceReload)
    {
        $results = [];
        $orderShippings = $this->getOrderShipping($order_id, $forceReload);
        $orderShippingsDetails = $this->getOrderShippingDetails($order_id, $forceReload);

        foreach($orderShippings as $shipping) {
            $shippingDetails = array_filter($orderShippingsDetails, function($item) use ($shipping){
                return $item['expedition_id'] == $shipping['expedition_id'];
            });
            $shipping['pdf_link'] = TW_Shipping::tw_pdf_link(
                $shipping['expedition_type'],
                $shipping['expedition_id'],
                $shipping['order_id']
            );
            $shipping['shipping_details'] = $shippingDetails;
            $results[] = $shipping;
        }
        return $results;
    }

    private function getOrderShipping($order_id, $forceReload = false){
        global $wpdb;

        if(!isset($this->_cache['shipping'][$order_id]) || $forceReload === true){
            $query = "SELECT * FROM tw_shipping where order_id = " . esc_sql($order_id);
            $res = $wpdb->get_results(
                $query,
                ARRAY_A
            );
            $this->_cache['shipping'][$order_id] = $res;
        }
        return isset($this->_cache['shipping'][$order_id]) ? $this->_cache['shipping'][$order_id] : [];
    }

    private function getShippingDetailsByExpeditionId($expeditionId){
        global $wpdb;

        $query = "SELECT twsd.* 
        FROM tw_shipping_detail twsd
        where twsd.expedition_id = " . esc_sql($expeditionId);
        
        $res = $wpdb->get_results(
            $query,
            ARRAY_A
        );
        return empty($res) ? null : $res;
    }
    
    private function getOrderShippingDetails($order_id, $forceReload = false){
        global $wpdb;

        if(!isset($this->_cache['shippingDetail'][$order_id]) || $forceReload === true){
            $query = "SELECT twsd.* 
            FROM tw_shipping tws
            INNER JOIN tw_shipping_detail twsd ON (twsd.expedition_id = tws.expedition_id)
            where tws.order_id = " . esc_sql($order_id);
            
            $res = $wpdb->get_results(
                $query,
                ARRAY_A
            );
            $this->_cache['shippingDetail'][$order_id] = $res;
        }
        return isset($this->_cache['shippingDetail'][$order_id]) ? $this->_cache['shippingDetail'][$order_id] : [];
    }

    public function get_shipping_zone_id_from_order($order_id) {
        // Charger la commande
        $order = wc_get_order($order_id);
    
        if (!$order) {
            return null; // Commande invalide
        }
    
        // Récupérer les informations de destination de la commande
        $destination = array(
            'country'   => $order->get_shipping_country(),
            'state'     => $order->get_shipping_state(),
            'postcode'  => $order->get_shipping_postcode(),
            'city'      => $order->get_shipping_city(),
            'address'   => $order->get_shipping_address_1(),
            'address_2' => $order->get_shipping_address_2(),
        );
    
        // Créer un package basé sur les informations de destination
        $package = array(
            'destination' => $destination,
            'contents'    => array(), // Vous pouvez ajouter les articles ici si nécessaire
        );
    
        // Trouver la zone d'expédition correspondante
        $shipping_zone = WC_Shipping_Zones::get_zone_matching_package($package);
    
        if ($shipping_zone) {
            return $shipping_zone->get_id(); // Retourne l'ID de la zone d'expédition
        }
    
        return null; // Aucune zone correspondante trouvée
    }

    public function get_shipping_methods_from_order($order_id)
    {
        $shipping_zone_id = $this->get_shipping_zone_id_from_order($order_id);
        $shipping_zone = new WC_Shipping_Zone($shipping_zone_id);
        return $shipping_zone->get_shipping_methods(true);
    }

    public function insert_shipping(&$shipment)
    {
        global $wpdb;
        $wpdb->insert(
            'tw_shipping',
            $shipment->to_array()
        );

        $shipment->expedition_id = $wpdb->insert_id;
        return $shipment->expedition_id;
    }

    public function update_shipping(&$shipment)
    {
        global $wpdb;

        $wpdb->update(
            'tw_shipping',
            $shipment->to_array(),
            ['expedition_id' => $shipment->expedition_id]
        );
    }

    public function insert_shipping_details($shippingDetailValues)
    {
        global $wpdb;
        foreach($shippingDetailValues as $item) {
            $wpdb->insert(
                'tw_shipping_detail',
                $item->to_array()
            );
        }
    }

    public function delete_shipping($shipment)
    {
        global $wpdb;
        $wpdb->delete(
            'tw_shipping',
            array(
                'expedition_id' => $shipment->expedition_id
            )
        );
    }

    public function delete_shipping_details($shipment)
    {
        global $wpdb;
        $wpdb->delete(
            'tw_shipping_detail',
            array(
                'expedition_id' => $shipment->expedition_id
            )
        );
    }
}