<?php

defined( 'ABSPATH' ) || exit;

/**
 * utiliser setassign
 * https://www.setasign.com/products/fpdi/about/ pour merger les pdf
 * voir les options dans lpc_label_print_action
 */
abstract class TW_Shipping_Handler {
    
    private $tmpShipmentDetails = null;

    public function canHandle(TW_Shipping $shipping)
    {
        return false;
    }

    public function generatePdf(TW_Shipping $shipping, $shipmentDetails, $outputMode = 'I')
    {

    }

    protected function _getPreviousShippingMethod(WC_Order $order)
    {
        $previousShippingMethod = null;
        if(count($order->get_shipping_methods()) > 0) {
            $shipping_method_keys = array_keys($order->get_shipping_methods());
            $previousShippingMethod = $order->get_shipping_methods()[$shipping_method_keys[0]];
        }

        return $previousShippingMethod;
    }

    protected function _setNewShippingMethod(TW_Shipping $shipping, $previousShippingMethod)
    {
        $newShippingMethod = clone $previousShippingMethod;
        list($carrierMethodId, $carrierInstanceId) = explode(':', $shipping->carrier);
        
        $newShippingMethod->set_method_id($carrierMethodId);
        $newShippingMethod->set_instance_id($carrierInstanceId);
        $newShippingMethod->save();
    }

    protected function _restorePreviousShippingMethod($previousShippingMethod)
    {
        $previousShippingMethod->set_instance_id($previousShippingMethod->get_instance_id());
        $previousShippingMethod->set_method_id($previousShippingMethod->get_method_id());
        $previousShippingMethod->save();
    }

    protected function _getItemsListAndInsuranceAmount(TW_Shipping $shipping, $shipmentDetails)
    {
        $assuranceAmount = 0;
        $items = [];

        foreach($shipmentDetails as $detail) {
            $items[$detail->item_id] = [
                'price' => $shipping->format_number($detail->item_price, 2),
                'qty' => (int)$detail->item_quantity,
                'weight' => $shipping->format_number($detail->item_weight)
            ];

            $itemValue = $detail->item_quantity * (float)$detail->item_price;
            $assuranceAmount = $assuranceAmount + $itemValue;
        }

        return [$items, $assuranceAmount];
    }

    public function isOrderFullyShipped(WC_Order $order) : bool
    {
        $nbrFullyShippedArticles = 0;
        $shipingRepository = TW_Factory::getInstance()->getClass(TW_Shipping_Repository::class);

        $orderedArticles = $order->get_items('line_item');
        foreach($orderedArticles as $article) {
            $orderedQuantity = (int)$article->get_quantity();
            $shippedQuantity = $shipingRepository->getShippedQuantityForOrderItem($order->get_id(), $article->get_id(), true);
            if($orderedQuantity - $shippedQuantity < 1) {
                $nbrFullyShippedArticles++;
            }
        }

        return $nbrFullyShippedArticles == count($orderedArticles) ? true : false;
    }

    public function generateShipmentTrackingFromShipmentInfo(TW_Shipping $shipping, $shipmentDetails, $useInsurance)
    {
    }

    public function getGeneratedLabelFilename(WC_Order $order)
    {
        $upload_dir = wp_upload_dir();
        $upload_dir = $upload_dir['basedir'];
        if(!file_exists($upload_dir . '/tmp')){
            mkdir($upload_dir . '/tmp', 0740);
        }

        $filename = $upload_dir . '/tmp/' . 'tmp_generated_label_' . $order->get_id() . '.pdf';
        return $filename;
    }

    /**
     * TODO faire un cron pour nettoyer le dossier temporaire
     */
    public function getArticlesListFilename(WC_Order $order)
    {
        $upload_dir = wp_upload_dir();
        $upload_dir = $upload_dir['basedir'];
        if(!file_exists($upload_dir . '/tmp')){
            mkdir($upload_dir . '/tmp', 0740);
        }

        $filename = $upload_dir . '/tmp/' . 'tmp_articles_list_' . $order->get_id() . '.pdf';
        return $filename;
    }

    public function generateSentArticlesList(TW_Shipping $shipping, $shipmentDetails, $outputMode = 'I')
    {
        $order = wc_get_order($shipping->order_id);
        $this->tmpShipmentDetails = $shipmentDetails;
        
        $filename = $this->getArticlesListFilename($order);

        $pdf = new TCPDF();
        $pdf->SetMargins(15, 5, 15);
        $pdf->SetHeaderMargin(5);
        $pdf->SetFooterMargin(10);
        $pdf->SetPrintHeader(false);
        $pdf->SetPrintFooter(false);
        $pdf->addPage();
        $pdf->writeHTML($this->printBlanklines(2));
        $pdf->writeHTML($this->printStoreInformationInHTML($order));
        $pdf->writeHTML($this->printBlanklines(2));
        $pdf->writeHTML($this->printOrderTitle($order));
        $pdf->writeHTML($this->printBlanklines(5));
        $pdf->writeHTML($this->printProductsTable($order));
        $pdf->writeHTML($this->printBlanklines(4));
        $pdf->writeHTML($this->printCustomerInformationInHTML($order));
        $pdf->Output($filename, $outputMode);

        $this->tmpShipmentDetails = null;
    }

    public function filterArticlesList($articles, $order, $types)
    {
        $shippedArticlesIds = [];
        $shippedArticles = [];

        if($this->tmpShipmentDetails !== null && is_array($this->tmpShipmentDetails)) {
            foreach($this->tmpShipmentDetails as $shipmentDetail) {
                if(!in_array($shipmentDetail->item_id, $shippedArticlesIds)) {
                    $shippedArticlesIds[] = $shipmentDetail->item_id;
                }
            }
        }

        foreach($articles as $item) {
            if(in_array($item->get_id(), $shippedArticlesIds))
            {
                $shipmentDetail = tw_array_where($this->tmpShipmentDetails, 'item_id', function($detail) use ($item){
                    return $detail->item_id == $item->get_id(); 
                });
                
                if($shipmentDetail !== null) {
                    $quantity =  $shipmentDetail->item_quantity;

                    $newArticle = clone $item;
                    $newArticle->set_quantity($quantity);
                    $shippedArticles[] = $newArticle;
                }
            }
        }

        return $shippedArticles;
    }

    public function printBlanklines($nbLines) {
        $output = '';
        for ($i = 0; $i < $nbLines; $i ++) {
            $output .= '<br/>';
        }

        return $output;
    }

    public function printOrderTitle($order) {
        $output = '<h3>' . sprintf(__('[Order #%s]', 'tw-core'), $order->get_id()) . ' (' . wc_format_datetime($order->get_date_created()) . ')</h3>';

        return $output;
    }

    protected function printStoreInformationInHTML(WC_Order $order) {
        $store_address2 = '';
        if (get_option('woocommerce_store_address_2')) {
            $store_address2 = get_option('woocommerce_store_address_2') . '<br>';
        }

        $store_name = defined('STORE_COMPANYNAME') ? STORE_COMPANYNAME : '';
        $store_address = defined('STORE_STREET') ? STORE_STREET : '';
        $store_address2 = defined('STORE_STREET2') ? STORE_STREET2 : '';
        $store_zipcode = defined('STORE_ZIPCODE') ? STORE_ZIPCODE : '';
        $store_city = defined('STORE_CITY') ? STORE_CITY : '';
        $store_countrycode = defined('STORE_COUNTRYCODE') ? STORE_COUNTRYCODE : '';

        $output_html = '<div><span style="font-weight: bold;font-size: 1.2em;">' . $store_name . '</span><br>';
        $output_html .= $store_address . ' ' . $store_zipcode . ' <br>';
        $output_html .= $store_city . ' <br>';
        $output_html .= WC()->countries->countries[$store_countrycode] . ' <br>';
        return $output_html;
    }

    protected function printProductsTable(WC_Order $order){
        add_filter('woocommerce_order_get_items', array($this, 'filterArticlesList'), PHP_INT_MAX, 3);
        $output = '';

        try {
            // Table header
            $output = '<div style="margin-bottom: 40px;">';
            $output .= '<table class="td" cellspacing="0" cellpadding="6" style="width: 100%; font-family: \'Helvetica Neue\', Helvetica, Roboto, Arial, sans-serif;" border="1">';
            $output .= '<thead><tr style="font-weight: bold; background-color: #eeeeee">';
            $output .= '<th class="td" scope="col" style="text-align:left;">' . __('Product', 'wc_colissimo') . '</th>';
            $output .= '<th class="td" scope="col" style="text-align:left;">' . __('Quantity', 'wc_colissimo') . '</th>';
            $output .= '<th class="td" scope="col" style="text-align:left;">' . __('Price', 'wc_colissimo') . '</th>';
            $output .= '</tr></thead>';

            // Products using email invoice structure
            $output .= '<tbody>';
            $output .= wc_get_email_order_items(
                $order,
                [
                    'show_sku'      => false,
                    'show_image'    => false,
                    'image_size'    => [32, 32],
                    'plain_text'    => '',
                    'sent_to_admin' => false,
                ]
            );

            $output .= '</tbody>';

            // Totals, payment and delivery methods
            $output .= '<tfoot>';
            
            if ($order->get_customer_note()) {
                $output .= '<tr>
                        <th class="td" scope="row" colspan="2" style="text-align:left;">Note:</th>
                        <td class="td" style="text-align:left;">' . wptexturize($order->get_customer_note()) . '</td>
                    </tr>';
            }
            $output .= '</tfoot></table></div>';
        } catch (Exception $ex) {}
        remove_filter('woocommerce_order_get_items', array($this, 'filterArticlesList'));

        return $output;
    }

    protected function printCustomerInformationInHTML(WC_Order $order) {
        $html_output      = '<table><tr>';
        $billing_address2 = '';
        if ($order->get_billing_address_2()) {
            $billing_address2 = $order->get_billing_address_2() . '<br>';
        }
        $shipping_address2 = '';
        if ($order->get_shipping_address_2()) {
            $shipping_address2 = $order->get_shipping_address_2() . '<br>';
        }

        // Billing address
        $html_output .= '<td><div>';
        $html_output .= '<span style="font-weight: bold;text-decoration: underline;">' . __('Billing address', 'wc_colissimo') . '</span><br>';
        if (!empty($order->get_billing_company())) {
            $html_output .= $order->get_billing_company() . '<br>';
        }
        $html_output .= $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() . '<br>';
        $html_output .= $order->get_billing_address_1() . '<br>';
        $html_output .= $billing_address2 . $order->get_billing_postcode() . ' <br>';
        $html_output .= $order->get_billing_city() . ' <br>';
        $html_output .= WC()->countries->countries[$order->get_billing_country()] . ' <br>';
        $html_output .= '</div></td>';
        $html_output .= '<td></td>';

        // Shipping address
        $html_output .= '<td><div style="float:right;">';
        $html_output .= '<span style="font-weight: bold;text-decoration: underline;">' . __('Shipping address', 'wc_colissimo') . '</span><br>';
        if (!empty($order->get_shipping_company())) {
            $html_output .= $order->get_shipping_company() . '<br>';
        }
        $html_output .= $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() . '<br>';
        $html_output .= $order->get_shipping_address_1() . '<br>';
        $html_output .= $shipping_address2 . $order->get_shipping_postcode() . ' <br>';
        $html_output .= $order->get_shipping_city() . ' <br>';
        $html_output .= WC()->countries->countries[$order->get_shipping_country()] . ' <br>';
        $html_output .= '</div></td>';

        $html_output .= '</tr></table>';

        return $html_output;
    }
}